/* ---------------------------------------------------------------------
 * Datasheet for mega48: http://www.atmel.com/images/doc2545.pdf
 * 
 * -------------------------------------------------------------------*/

//sudo avrdude -c usbasp -p m48 -u -U flash:w:main.hex:a
//stock fuses: lfuse=62, hfuse=DF, efuse=1
//int. rx osc 8mhz  ckdiv8 = 1mhz
//flashen: sudo avrdude -c usbasp -p m48 -u -U flash:w:main.hex:a


//#define F_CPU 1000000UL  //already set in makefile



#include <avr/io.h>  
#include <util/delay.h>
#include <stdint.h>

#define CTC_MATCH_OVERFLOW ((F_CPU / 1000) / 8) 
#include <avr/interrupt.h>
#include <util/atomic.h>

#define BUTTON_START PC0
#define BUTTON_UP PC1
#define BUTTON_DOWN PC2

#define PIN_RELAIS PC5

#define PIN_NTC PC3

/*
*Wiring segements
* --a--
* f   b
* - g - 
* e   c
* --d-- -DP
*#############
* --0--
* 5   1
* - 6 - 
* 4   2
* --3-- -7

*
*/

const uint8_t segmap[] = { //bit 0=segment on
0b11000000, //0
0b11111001, //1
0b10100100, //2
0b10110000, //3
0b10011001, //4
0b10010010, //5
0b10000010, //6
0b11111000, //7
0b10000000, //8
0b10010000, //9
0b01111111, //.
0b11111111  // off
};


//variables
unsigned long timeset;
unsigned long timeleft;
uint8_t running;

uint16_t timeupdownButtondelay;
#define TIMEUPDOWNBUTTONDELAY_START 200
#define TIMEUPDOWNBUTTONDELAY_STEP 10 //subtract from delay if long pressed
#define TIMEUPDOWNBUTTONDELAY_MIN 50 //minimum delay

#define MAXTIMESET 99000
#define MINTIMESET 100


//Temperature controller variables
uint16_t tempset=200; //x10,  200= 20degC
#define MAXTEMPSET 990
#define MINTEMPSET 0
uint8_t tempcontrolMode=0; //0=inactive, 1=active, 2=P setup, 3=I setup
uint8_t heatingPWM=0;  //slow pwm value

uint8_t regulationMode=0; //0=heating, 1=cooling regulation

uint8_t Kp=32; //Proportional
uint8_t Ki=8; //Integral
uint16_t tempdiffIntegral=32768; //offset for negative values
#define TEMPDIFFINTEGRALMIN 1000 //1000 (maximum one step can increase
#define TEMPDIFFINTEGRALMAX 64536 //2^16 - 1000

#define STEPDELAY 100
unsigned long lastStepdelay=0;
uint8_t showdecimalcounter=0;
#define MAX_SHOWDECIMALCOUNTER 30 //count decimalcounter up to..
#define SHOWDECIMALAFTERCOUNTER 20 //0<x<MAX_SHOWDECIMALCOUNTER.  the closer to MAX_SHOWDECIMALCOUNTER the shorter decimal is shown
uint8_t showtempdecimal=0; //if 1, then show digit after decimal point

uint8_t heatingcounter=0;


volatile unsigned long timer1_millis;
 
ISR (TIMER1_COMPA_vect)
{
    timer1_millis++;
}

void resetTimer1(void){
    timer1_millis=0;
}

unsigned long millis (void)
{
    unsigned long millis_return;

    // Ensure this cannot be disrupted
    ATOMIC_BLOCK(ATOMIC_FORCEON) {
        millis_return = timer1_millis;
    }
 
    return millis_return;
}


void displayOff(void){
	PORTB = segmap[11];
	PORTD = segmap[11];
}

void disable(void){ //turn lamp off
	running=0; //turn off
	PORTC &= ~(1 << PIN_RELAIS);
}
void enable(void){ //turn lamp on
	running=1;
	PORTC |= (1 << PIN_RELAIS);
}

void delay_ms_loop(unsigned long t){ //for variable time value
	for (unsigned long i=0;i<t;i++){
		_delay_ms(1);
	}
}


uint16_t roundFloatToInt(float myfloat)
{
  double integral;
  float fraction = (float)modf(myfloat, &integral);
 
  if (fraction >= 0.5)
    integral += 1;
  if (fraction <= -0.5)
    integral -= 1;
 
  return (uint16_t)integral;
}

/*float getTemp(float T0, float R0, float B, float RV, float VA_VB)
{
  T0+=273.15;  // umwandeln Celsius in absolute Temperatur
  float RN=RV*VA_VB / (1-VA_VB); // aktueller Widerstand des NTC, bei anschluss des NTC nach masse
  return T0 * B / (B + T0 * log(RN / R0))-273.15;
}*/

uint16_t getTempEfficient(float VA_VB) //flash efficient version of above function for specific values
{ //T0=25.0f,R0=10000.0f,B=3435.0f,RV=10000.0f,ADC/1023.0
  // umwandeln Celsius in absolute Temperatur,  T0=25.0f + 273.15
  float RN=10000.0f*VA_VB / (1-VA_VB); // aktueller Widerstand des NTC, bei anschluss des NTC nach masse
  //return roundFloatToInt( 10*    298.15f * 3435.0f / (3435.0f + 298.15f * log(RN / 10000))-273.15 );
  return roundFloatToInt( (298.15f * 3435.0f  / (3435.0f +   298.15f * log(RN / 10000.0f))-273.15  )*10);
}




int main(void)
{
	timeset=2000; //in ms
	timeleft=timeset;
	running=0;

	timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_START;

	uint8_t seg10=0xFF; //store display
	uint8_t seg1=0xFF;
	
	DDRB = 0xFF; //7segments 10er, all output
	DDRD = 0xFF; //7segments 1er, all output
	DDRC = 0x00; DDRC |= (1<< PIN_RELAIS); //pullup //all inputs, PIN_RELAIS output

	PORTC |= (1<< BUTTON_START); //pullup
	PORTC |= (1<< BUTTON_UP); //pullup
	PORTC |= (1<< BUTTON_DOWN); //pullup


	//ADC
	ADMUX = (1<<REFS0);               // Set Reference to AVCC
	ADMUX |= (1 << MUX0);
	ADMUX |= (1 << MUX1); //mux 2..0  011  = adc3

	ADCSRA = (1<<ADEN)|(1<<ADPS2);    // Enable ADC, set prescaler to 16
                                      // Fadc=Fcpu/prescaler=1000000/16=62.5kHz
                                      // Fadc should be between 50kHz and 200kHz
                                      // Enable ADC conversion complete interrupt


	ADCSRA |= (1<<ADSC);              // Start the first conversion


	//Timer and interrupts
	// CTC mode, Clock/8
	TCCR1B |= (1 << WGM12) | (1 << CS11);
	 
	// Load the high byte, then the low byte
	// into the output compare
	OCR1AH = (CTC_MATCH_OVERFLOW >> 8);
	OCR1AL = CTC_MATCH_OVERFLOW;
	 
	// Enable the compare match interrupt
	TIMSK1 |= (1 << OCIE1A);

	sei();

	//Setup done

	/*for (uint8_t i=0;i<12;i++){ //TEST
		enable();
		_delay_ms(1000);
		PORTD = segmap[i];
		PORTB = segmap[i];
	}
	disable();*/

	/*while (1)
	{
		
		if (!(PINC & (1<<BUTTON_START))){
			_delay_ms(1000);
			PORTD = segmap[1];
		}else{
			_delay_ms(1000);
			PORTD = segmap[11];
		}
	}*/


  	_delay_ms(100);

  if ((PINC & (1<<BUTTON_UP)) && (PINC & (1<<BUTTON_DOWN)) ){  //Timer mode, if buttons not pressed at boot
	while(42)
	{

		//Buttons
		if (!(PINC & (1<<BUTTON_START))){ //pin low = button pressed
			if (running==0) //isnt already running
			{
				while(!(PINC & (1<<BUTTON_START))){ //debounce
					_delay_ms(100);
				}
				/*uint16_t loadstepdelay=timeset/20; //faster animation if short on-time
				//loading animation
				PORTB=0b11111111; PORTD=0b11111001; //right
				delay_ms_loop(loadstepdelay);
				PORTB=0b11111111; PORTD=0b11001111; //second right
				delay_ms_loop(loadstepdelay);
				PORTB=0b11111001; PORTD=0b11111111; //second left
				delay_ms_loop(loadstepdelay);
				PORTB=0b11001111; PORTD=0b11111111; //left
				delay_ms_loop(loadstepdelay);*/

				//delay_ms_loop(timeset<<2); //time dependent wait
				//_delay_ms(500); //simple wait

				//loading animation
				PORTB=0b11111111; PORTD=0b11111001; //right
				_delay_ms(150);
				PORTB=0b11111111; PORTD=0b11001111; //second right
				_delay_ms(150);
				PORTB=0b11111001; PORTD=0b11111111; //second left
				_delay_ms(150);
				PORTB=0b11001111; PORTD=0b11111111; //left
				_delay_ms(150);


				resetTimer1();
				enable();
			}
		}

		if (running==0)
		{
			if (!(PINC & (1<<BUTTON_UP))){
				delay_ms_loop(timeupdownButtondelay);
				timeupdownButtondelay-=TIMEUPDOWNBUTTONDELAY_STEP;
				if (timeupdownButtondelay<TIMEUPDOWNBUTTONDELAY_MIN){ //bound
					timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_MIN;
				}

				if (timeset<10000){
					timeset+=100;
				}else{
					timeset+=1000;
				}
				if (timeset>MAXTIMESET){ //bound
					timeset=MAXTIMESET;
				}

			}else if (!(PINC & (1<<BUTTON_DOWN))){
				delay_ms_loop(timeupdownButtondelay);
				timeupdownButtondelay-=TIMEUPDOWNBUTTONDELAY_STEP;
				if (timeupdownButtondelay<TIMEUPDOWNBUTTONDELAY_MIN){ //bound
					timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_MIN;
				}

				if (timeset<=10000){
					timeset-=100;
				}else{
					timeset-=1000;
				}
				if (timeset<MINTIMESET){ //bound
					timeset=MINTIMESET;
				}
			}else{
				timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_START; //reset delay
			}
		}else{ //running==1
			if ( !(PINC & (1<<BUTTON_UP)) || !(PINC & (1<<BUTTON_DOWN)) ){ //is running, up or down pressed
				disable(); //stop enabled
				displayOff();
				//_delay_ms(1000);
			}
		}


		//timestuff
		if (running==1){
			unsigned long _millis=millis();
			timeleft=timeset-_millis; //calculate timeleft for display
			if (_millis>timeset){ //time is over
				disable();
			}
		}else{
			timeleft=timeset; //show settime
		}
	
		//display time
		if (timeleft<10000){ //then display decimals
			uint8_t _seconds=timeleft/1000;
			uint8_t _tenth=(timeleft%1000)/100;
			seg10 = segmap[_seconds];
			seg10 &= ~0b10000000; //show dot
			seg1 = segmap[_tenth];
		}else{ //only full seconds
			uint8_t _seconds1=(timeleft/1000)%10;
			uint8_t _seconds10=timeleft/10000;
			seg10 = segmap[_seconds10];
			seg1 = segmap[_seconds1];
		}
		

		//set 7segment port outputs
		PORTB = seg10;
		PORTD = seg1;
	}

  }else{ //temperature controller

	while(!(PINC & (1<<BUTTON_UP))) {  //wait for button release, set heating mode
		//_delay_ms(100); 
		//regulationMode=0;  //is already init. with 0
		PORTD=0b11111110; //up bar
	}
	while(!(PINC & (1<<BUTTON_DOWN))) {  //wait for button release, set cooling mode
		//_delay_ms(100); 
		regulationMode=1;
		PORTD=0b11110111; //bottom bar
	}

	_delay_ms(500);

	uint8_t blink=0;

	uint16_t tempDeg=0;

	while(42){
		ADCSRA |= (1<<ADSC);
		while(ADCSRA &(1<<ADSC));     // Wait until conversion is finished
		//float ntc_temp=getTemp(25.0f,10000.0f,3435.0f,10000.0f,ADC/1023.0); //10k ntc with B=3435 and 10k pullup  http://www.pollin.de/shop/downloads/D220757D.PDF
		

		

		//change active,inactive with start_button
		if (!(PINC & (1<<BUTTON_START))){ //pin low = button pressed
			while(!(PINC & (1<<BUTTON_START))); //wait release
			if (tempcontrolMode>1){ //was in setup mode
				tempcontrolMode=1; //set to active, cause was active when went to setup
			}else{
				tempcontrolMode=1-tempcontrolMode;
			}
			_delay_ms(200);
		}



		if (millis()-lastStepdelay>STEPDELAY){
			blink++;
			blink%=3;
		
			showdecimalcounter++;
			showdecimalcounter%=MAX_SHOWDECIMALCOUNTER;
			

			heatingcounter++;//overflows at 256, for slow software pwm

			if (heatingcounter%10==0){ //for slower calculation and regulation

				//tempDeg=roundFloatToInt(ntc_temp*10); //degC x10 //get new temperature
				tempDeg=getTempEfficient(ADC/1023.0); //shrinked version of getTemp

				if (tempcontrolMode==0){ //PI Controller active
					tempdiffIntegral=32768; //reset Integral
					heatingPWM=0; //output off
				}else{
					uint16_t tempdiff=32768; 
					if (regulationMode==0){ //heating regulation
						tempdiff=(tempdiff+tempset)-tempDeg;//if >32768, then needs heating (higher PWM)
					}else{
						tempdiff=(tempdiff+tempDeg)-tempset;//if >32768, then needs cooling  (higher PWM)
					}
					
					//Integral anti wind-up, freeze integration if output PWM limits reached
					if ((tempdiff>32768 && heatingPWM<255) || (tempdiff<32768 && heatingPWM>0)){ //needs higher pwm output and upper limit not reached OR needs lower PWM output and lower limit not reached
						tempdiffIntegral=(tempdiff-32768)+tempdiffIntegral ; 
					}

					
					/*if (tempdiffIntegral<TEMPDIFFINTEGRALMIN){ //not needed with anti wind-up
						tempdiffIntegral=TEMPDIFFINTEGRALMIN;
					}else if (tempdiffIntegral>TEMPDIFFINTEGRALMAX){
						tempdiffIntegral=TEMPDIFFINTEGRALMAX;
					}*/

					uint16_t _setValue=32768+ (Kp*(tempdiff-(32768)))+( (tempdiffIntegral-(32768)>>4)* Ki );  //>>4=div 16
					if (_setValue<(32768)){ //negative
						_setValue=0;
					}else if (_setValue>33023){ //2^15+255
						_setValue=255;
					}else{ //value is within bounds
						_setValue=_setValue-(32768);
					}
					heatingPWM=_setValue;
					
				}
			}


			lastStepdelay=millis();
		}


		
		if (!(PINC & (1<<BUTTON_UP))){
			_delay_ms(100);

			if(tempcontrolMode==0){
				tempset++;
			
				if (tempset>MAXTEMPSET){ //bound
					tempset=MAXTEMPSET;
				}
			}else if(tempcontrolMode==1){ //up pressed and tempcontroller active
				uint8_t _holdcounter=0;
				while(!(PINC & (1<<BUTTON_UP))){
					_holdcounter++;
					_delay_ms(50); //steptime 50->  1sec=20steps
				}
				if (_holdcounter>40){ //2secs hold
					tempcontrolMode=2; //P setup
				}
			}else if(tempcontrolMode==2){ //P setup
				Kp++;
			}else if(tempcontrolMode==3){ //I setup
				Ki++;
			}

		}else if (!(PINC & (1<<BUTTON_DOWN))){
			_delay_ms(100);

			if(tempcontrolMode==0){
				tempset-=1;
			
				if (tempset<MINTEMPSET){ //bound
					tempset=MINTEMPSET;
				}
			}else if(tempcontrolMode==1){ //down pressed and tempcontroller active
				uint8_t _holdcounter=0;
				while(!(PINC & (1<<BUTTON_DOWN))){
					_holdcounter++;
					_delay_ms(50); //steptime 50->  1sec=20steps
				}
				if (_holdcounter>40){ //2secs hold
					tempcontrolMode=3; //I setup
				}
			}else if(tempcontrolMode==2){ //P setup
				Kp--;
			}else if(tempcontrolMode==3){ //I setup
				Ki--;
			}
		}
			



		//show temperature
		uint16_t tempdispl;
		if(tempcontrolMode==0){
			tempdispl=tempset;
		}else if(tempcontrolMode==1){
			tempdispl=tempDeg;
		}else if(tempcontrolMode==2){ //P setup
			tempdispl=Kp;
		}else if(tempcontrolMode==3){ //I setup
			tempdispl=Ki;
		}

		if(blink>=1 || tempcontrolMode==1){
			if (showdecimalcounter<SHOWDECIMALAFTERCOUNTER){
				uint8_t _temp1=(tempdispl/10)%10;
				uint8_t _temp10=tempdispl/100;
				seg10 = segmap[_temp10];
				seg1 = segmap[_temp1];
			}else{ //show digit after decimal point
				uint8_t _tempdecimal=(tempdispl)%10;
				seg10 = segmap[10]; //dot
				seg1 = segmap[_tempdecimal];
			}
		}else{
			seg10 = segmap[11]; //off
			seg1 = segmap[11]; //off
		}



		if (heatingcounter<heatingPWM){ //slow software pwm
			seg1&= segmap[10];
			enable();
		}else{
			disable();
		}
		
		
		//set 7segment port outputs
		PORTB = seg10;
		PORTD = seg1;

	}

  }

	return (0);
}




