/* ---------------------------------------------------------------------
 * Datasheet for mega48: http://www.atmel.com/images/doc2545.pdf
 * 
 * -------------------------------------------------------------------*/

//sudo avrdude -c usbasp -p m48 -u -U flash:w:main.hex:a
//stock fuses: lfuse=62, hfuse=DF, efuse=1
//int. rx osc 8mhz  ckdiv8 = 1mhz
//flashen: sudo avrdude -c usbasp -p m48 -u -U flash:w:main.hex:a


//#define F_CPU 1000000UL  //already set in makefile



#include <avr/io.h>  
#include <util/delay.h>
#include <stdint.h>

#define CTC_MATCH_OVERFLOW ((F_CPU / 1000) / 8) 
#include <avr/interrupt.h>
#include <util/atomic.h>

#define BUTTON_START PC0
#define BUTTON_UP PC1
#define BUTTON_DOWN PC2

#define PIN_RELAIS PC5

/*
*Wiring segements
* --a--
* f   b
* - g - 
* e   c
* --d-- -DP
*#############
* --0--
* 5   1
* - 6 - 
* 4   2
* --3-- -7

*
*/

const uint8_t segmap[] = { //bit 0=segment on
0b11000000, //0
0b11111001, //1
0b10100100, //2
0b10110000, //3
0b10011001, //4
0b10010010, //5
0b10000010, //6
0b11111000, //7
0b10000000, //8
0b10010000, //9
0b01111111, //.
0b11111111  // off
};


//variables
unsigned long timeset;
unsigned long timeleft;
uint8_t running;

uint16_t timeupdownButtondelay;
#define TIMEUPDOWNBUTTONDELAY_START 200
#define TIMEUPDOWNBUTTONDELAY_STEP 10 //subtract from delay if long pressed
#define TIMEUPDOWNBUTTONDELAY_MIN 50 //minimum delay

#define MAXTIMESET 60000
#define MINTIMESET 100


volatile unsigned long timer1_millis;
 
ISR (TIMER1_COMPA_vect)
{
    timer1_millis++;
}

void resetTimer1(void){
    timer1_millis=0;
}

unsigned long millis (void)
{
    unsigned long millis_return;

    // Ensure this cannot be disrupted
    ATOMIC_BLOCK(ATOMIC_FORCEON) {
        millis_return = timer1_millis;
    }
 
    return millis_return;
}


void displayOff(void){
	PORTB = segmap[11];
	PORTD = segmap[11];
}

void disable(void){ //turn lamp off
	running=0; //turn off
	PORTC &= ~(1 << PIN_RELAIS);
}
void enable(void){ //turn lamp on
	running=1;
	PORTC |= (1 << PIN_RELAIS);
}

void delay_ms_loop(unsigned long t){
	for (unsigned long i=0;i<t;i++){
		_delay_ms(1);
	}
}


int main(void)
{
	timeset=2000; //in ms
	timeleft=timeset;
	running=0;

	timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_START;

	uint8_t seg10=0xFF; //store display
	uint8_t seg1=0xFF;
	
	DDRB = 0xFF; //7segments 10er, all output
	DDRD = 0xFF; //7segments 1er, all output
	DDRC = 0x00; DDRC |= (1<< PIN_RELAIS); //pullup //all inputs, PIN_RELAIS output

	PORTC |= (1<< BUTTON_START); //pullup
	PORTC |= (1<< BUTTON_UP); //pullup
	PORTC |= (1<< BUTTON_DOWN); //pullup

	//Timer and interrupts
	// CTC mode, Clock/8
	TCCR1B |= (1 << WGM12) | (1 << CS11);
	 
	// Load the high byte, then the low byte
	// into the output compare
	OCR1AH = (CTC_MATCH_OVERFLOW >> 8);
	OCR1AL = CTC_MATCH_OVERFLOW;
	 
	// Enable the compare match interrupt
	TIMSK1 |= (1 << OCIE1A);

	sei();

	//Setup done

	/*for (uint8_t i=0;i<12;i++){ //TEST
		enable();
		_delay_ms(1000);
		PORTD = segmap[i];
		PORTB = segmap[i];
	}
	disable();*/

	/*while (1)
	{
		
		if (!(PINC & (1<<BUTTON_START))){
			_delay_ms(1000);
			PORTD = segmap[1];
		}else{
			_delay_ms(1000);
			PORTD = segmap[11];
		}
	}*/


	while(42)
	{

		//Buttons
		if (!(PINC & (1<<BUTTON_START))){ //pin low = button pressed
			if (running==0) //isnt already running
			{
				while(!(PINC & (1<<BUTTON_START))){ //debounce
					_delay_ms(100);
				}
				uint16_t loadstepdelay=timeset/20; //faster animation if short on-time
				//loading animation
				PORTB=0b11111111; PORTD=0b11111001; //right
				delay_ms_loop(loadstepdelay);
				PORTB=0b11111111; PORTD=0b11001111; //second right
				delay_ms_loop(loadstepdelay);
				PORTB=0b11111001; PORTD=0b11111111; //second left
				delay_ms_loop(loadstepdelay);
				PORTB=0b11001111; PORTD=0b11111111; //left
				delay_ms_loop(loadstepdelay);
				resetTimer1();
				enable();
			}
		}

		if (running==0)
		{
			if (!(PINC & (1<<BUTTON_UP))){
				delay_ms_loop(timeupdownButtondelay);
				timeupdownButtondelay-=TIMEUPDOWNBUTTONDELAY_STEP;
				if (timeupdownButtondelay<TIMEUPDOWNBUTTONDELAY_MIN){ //bound
					timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_MIN;
				}

				if (timeset<10000){
					timeset+=100;
				}else{
					timeset+=1000;
				}
				if (timeset>MAXTIMESET){ //bound
					timeset=MAXTIMESET;
				}

			}else if (!(PINC & (1<<BUTTON_DOWN))){
				delay_ms_loop(timeupdownButtondelay);
				timeupdownButtondelay-=TIMEUPDOWNBUTTONDELAY_STEP;
				if (timeupdownButtondelay<TIMEUPDOWNBUTTONDELAY_MIN){ //bound
					timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_MIN;
				}

				if (timeset<=10000){
					timeset-=100;
				}else{
					timeset-=1000;
				}
				if (timeset<MINTIMESET){ //bound
					timeset=MINTIMESET;
				}
			}else{
				timeupdownButtondelay=TIMEUPDOWNBUTTONDELAY_START; //reset delay
			}
		}else{ //running==1
			if ( !(PINC & (1<<BUTTON_UP)) || !(PINC & (1<<BUTTON_DOWN)) ){ //is running, up or down pressed
				disable(); //stop enabled
				displayOff();
				_delay_ms(1000);
			}
		}


		//timestuff
		if (running==1){
			unsigned long _millis=millis();
			timeleft=timeset-_millis; //calculate timeleft for display
			if (_millis>timeset){ //time is over
				disable();
			}
		}else{
			timeleft=timeset; //show settime
		}
	
		//display time
		if (timeleft<10000){ //then display decimals
			uint8_t _seconds=timeleft/1000;
			uint8_t _tenth=(timeleft%1000)/100;
			seg10 = segmap[_seconds];
			seg10 &= ~0b10000000; //show dot
			seg1 = segmap[_tenth];
		}else{ //only full seconds
			uint8_t _seconds1=(timeleft/1000)%10;
			uint8_t _seconds10=timeleft/10000;
			seg10 = segmap[_seconds10];
			seg1 = segmap[_seconds1];
		}
		

		//set 7segment port outputs
		PORTB = seg10;
		PORTD = seg1;
	}

	return (0);
}

